<?php
/**
 * LUXOR IPTV - API Endpoint for App Configuration
 *
 * This endpoint is called by the Android app to get:
 * - List of Xtream servers (with fallback order)
 * - App branding settings
 *
 * Returns JSON response
 */

header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST');
header('Access-Control-Allow-Headers: Content-Type, Authorization');

require_once __DIR__ . '/../includes/db.php';

// Optional: Verify API key (uncomment for extra security)
// $apiKey = $_SERVER['HTTP_X_API_KEY'] ?? '';
// if ($apiKey !== API_SECRET_KEY) {
//     http_response_code(401);
//     echo json_encode(['error' => 'Unauthorized']);
//     exit;
// }

try {
    // Get active Xtream servers ordered by priority
    $servers = db()->fetchAll(
        "SELECT id, priority, name, host, port, protocol
         FROM xtream_servers
         WHERE is_active = 1
         ORDER BY priority ASC"
    );

    // Get app settings
    $settingsRaw = db()->fetchAll("SELECT setting_key, setting_value FROM app_settings");
    $settings = [];
    foreach ($settingsRaw as $row) {
        $settings[$row['setting_key']] = $row['setting_value'];
    }

    // Format servers for app
    $formattedServers = [];
    foreach ($servers as $server) {
        $formattedServers[] = [
            'id' => (int)$server['id'],
            'priority' => (int)$server['priority'],
            'name' => $server['name'],
            'host' => $server['host'],
            'port' => $server['port'],
            'protocol' => $server['protocol'],
            'base_url' => $server['protocol'] . '://' . $server['host'] .
                         ($server['port'] && $server['port'] !== '80' && $server['port'] !== '443'
                          ? ':' . $server['port'] : '')
        ];
    }

    // Log this connection (optional)
    $deviceId = $_GET['device_id'] ?? $_POST['device_id'] ?? null;
    if ($deviceId) {
        db()->insert('connection_logs', [
            'device_id' => $deviceId,
            'ip_address' => $_SERVER['REMOTE_ADDR'] ?? '',
            'user_agent' => $_SERVER['HTTP_USER_AGENT'] ?? '',
            'status' => 'success'
        ]);
    }

    // Response
    echo json_encode([
        'success' => true,
        'data' => [
            'servers' => $formattedServers,
            'settings' => [
                'app_name' => $settings['app_name'] ?? 'LUXOR IPTV',
                'app_logo_url' => $settings['app_logo_url'] ?? '',
                'primary_color' => $settings['primary_color'] ?? '#6200EE',
                'require_login' => ($settings['require_login'] ?? '1') === '1',
                'max_devices' => (int)($settings['max_devices_per_user'] ?? 3)
            ],
            'version' => $settings['panel_version'] ?? '1.0.0'
        ],
        'timestamp' => time()
    ], JSON_PRETTY_PRINT);

} catch (Exception $e) {
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'error' => 'Server error',
        'message' => $e->getMessage() // Remove in production
    ]);
}
