<?php
/**
 * LUXOR IPTV Admin Panel - Dashboard
 */

require_once __DIR__ . '/includes/auth.php';
Auth::require();

$user = Auth::user();

// Get servers
$servers = db()->fetchAll("SELECT * FROM xtream_servers ORDER BY priority ASC");

// Get connection stats
$stats = db()->fetch("
    SELECT
        COUNT(*) as total_connections,
        COUNT(DISTINCT device_id) as unique_devices,
        SUM(CASE WHEN status = 'success' THEN 1 ELSE 0 END) as successful,
        SUM(CASE WHEN status = 'failed' THEN 1 ELSE 0 END) as failed
    FROM connection_logs
    WHERE created_at >= DATE_SUB(NOW(), INTERVAL 24 HOUR)
");

// Get app settings
$settingsRaw = db()->fetchAll("SELECT setting_key, setting_value FROM app_settings");
$settings = [];
foreach ($settingsRaw as $row) {
    $settings[$row['setting_key']] = $row['setting_value'];
}

// Handle form submissions
$message = '';
$messageType = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $action = $_POST['action'] ?? '';

    try {
        switch ($action) {
            case 'save_server':
                $id = $_POST['server_id'] ?? null;
                $data = [
                    'priority' => (int)$_POST['priority'],
                    'name' => trim($_POST['name']),
                    'host' => trim($_POST['host']),
                    'port' => trim($_POST['port']) ?: '80',
                    'protocol' => $_POST['protocol'],
                    'is_active' => isset($_POST['is_active']) ? 1 : 0
                ];

                if ($id) {
                    db()->update('xtream_servers', $data, 'id = :id', ['id' => $id]);
                    $message = 'Server updated successfully';
                } else {
                    db()->insert('xtream_servers', $data);
                    $message = 'Server added successfully';
                }
                $messageType = 'success';
                break;

            case 'delete_server':
                $id = $_POST['server_id'] ?? null;
                if ($id) {
                    db()->delete('xtream_servers', 'id = :id', ['id' => $id]);
                    $message = 'Server deleted successfully';
                    $messageType = 'success';
                }
                break;

            case 'save_settings':
                foreach ($_POST['settings'] as $key => $value) {
                    db()->query(
                        "INSERT INTO app_settings (setting_key, setting_value)
                         VALUES (:key, :value)
                         ON DUPLICATE KEY UPDATE setting_value = :value2",
                        ['key' => $key, 'value' => $value, 'value2' => $value]
                    );
                }
                $message = 'Settings saved successfully';
                $messageType = 'success';
                break;

            case 'change_password':
                $current = $_POST['current_password'] ?? '';
                $new = $_POST['new_password'] ?? '';
                $confirm = $_POST['confirm_password'] ?? '';

                if (empty($current) || empty($new)) {
                    throw new Exception('All password fields are required');
                }
                if ($new !== $confirm) {
                    throw new Exception('New passwords do not match');
                }
                if (strlen($new) < 6) {
                    throw new Exception('Password must be at least 6 characters');
                }
                if (!Auth::changePassword($current, $new)) {
                    throw new Exception('Current password is incorrect');
                }

                $message = 'Password changed successfully';
                $messageType = 'success';
                break;
        }

        // Reload data after changes
        $servers = db()->fetchAll("SELECT * FROM xtream_servers ORDER BY priority ASC");
        $settingsRaw = db()->fetchAll("SELECT setting_key, setting_value FROM app_settings");
        $settings = [];
        foreach ($settingsRaw as $row) {
            $settings[$row['setting_key']] = $row['setting_value'];
        }

    } catch (Exception $e) {
        $message = $e->getMessage();
        $messageType = 'danger';
    }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Dashboard - <?php echo PANEL_NAME; ?></title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.1/font/bootstrap-icons.css" rel="stylesheet">
    <style>
        :root {
            --primary-color: #6200EE;
            --dark-bg: #1a1c1e;
            --card-bg: #2d2d3a;
        }
        body {
            background: var(--dark-bg);
            color: #fff;
            min-height: 100vh;
        }
        .sidebar {
            background: var(--card-bg);
            min-height: 100vh;
            padding: 20px;
            position: fixed;
            width: 250px;
        }
        .main-content {
            margin-left: 250px;
            padding: 30px;
        }
        .card {
            background: var(--card-bg);
            border: 1px solid rgba(255,255,255,0.1);
            border-radius: 12px;
        }
        .card-header {
            background: transparent;
            border-bottom: 1px solid rgba(255,255,255,0.1);
            color: #fff;
            font-weight: 600;
        }
        .stat-card {
            background: linear-gradient(135deg, var(--primary-color) 0%, #9c27b0 100%);
            border-radius: 12px;
            padding: 20px;
            color: #fff;
        }
        .stat-card.secondary {
            background: linear-gradient(135deg, #00bcd4 0%, #009688 100%);
        }
        .stat-card.success {
            background: linear-gradient(135deg, #4caf50 0%, #2e7d32 100%);
        }
        .stat-card.warning {
            background: linear-gradient(135deg, #ff9800 0%, #f57c00 100%);
        }
        .stat-number {
            font-size: 32px;
            font-weight: 700;
        }
        .nav-link {
            color: rgba(255,255,255,0.7);
            padding: 12px 16px;
            border-radius: 8px;
            margin-bottom: 4px;
        }
        .nav-link:hover, .nav-link.active {
            background: rgba(98, 0, 238, 0.2);
            color: #fff;
        }
        .nav-link i {
            margin-right: 10px;
        }
        .form-control, .form-select {
            background: rgba(255,255,255,0.1);
            border: 1px solid rgba(255,255,255,0.2);
            color: #fff;
        }
        .form-control:focus, .form-select:focus {
            background: rgba(255,255,255,0.15);
            border-color: var(--primary-color);
            color: #fff;
            box-shadow: 0 0 0 0.2rem rgba(98, 0, 238, 0.25);
        }
        .form-select option {
            background: var(--card-bg);
        }
        .btn-primary {
            background: var(--primary-color);
            border: none;
        }
        .btn-primary:hover {
            background: #7c3aed;
        }
        .server-card {
            background: rgba(255,255,255,0.05);
            border: 1px solid rgba(255,255,255,0.1);
            border-radius: 8px;
            padding: 16px;
            margin-bottom: 12px;
        }
        .server-card.active {
            border-color: #4caf50;
        }
        .server-card.inactive {
            opacity: 0.6;
        }
        .priority-badge {
            background: var(--primary-color);
            color: #fff;
            font-size: 12px;
            padding: 4px 8px;
            border-radius: 4px;
        }
        .logo {
            font-size: 32px;
            margin-bottom: 8px;
        }
        .brand-name {
            font-size: 18px;
            font-weight: 700;
            color: #fff;
        }
        .table {
            color: #fff;
        }
        .table-dark {
            --bs-table-bg: transparent;
        }
        .form-check-input:checked {
            background-color: var(--primary-color);
            border-color: var(--primary-color);
        }
    </style>
</head>
<body>
    <!-- Sidebar -->
    <div class="sidebar">
        <div class="text-center mb-4">
            <div class="logo">📺</div>
            <div class="brand-name"><?php echo PANEL_NAME; ?></div>
        </div>

        <nav class="nav flex-column">
            <a class="nav-link active" href="#servers" data-bs-toggle="tab">
                <i class="bi bi-hdd-stack"></i>Servers
            </a>
            <a class="nav-link" href="#settings" data-bs-toggle="tab">
                <i class="bi bi-gear"></i>Settings
            </a>
            <a class="nav-link" href="#stats" data-bs-toggle="tab">
                <i class="bi bi-bar-chart"></i>Statistics
            </a>
            <a class="nav-link" href="#account" data-bs-toggle="tab">
                <i class="bi bi-person"></i>Account
            </a>
        </nav>

        <div class="mt-auto pt-4 border-top border-secondary">
            <div class="text-muted small mb-2">
                <i class="bi bi-person-circle me-1"></i>
                <?php echo htmlspecialchars($user['username']); ?>
            </div>
            <a href="logout.php" class="btn btn-outline-danger btn-sm w-100">
                <i class="bi bi-box-arrow-left me-1"></i>Logout
            </a>
        </div>
    </div>

    <!-- Main Content -->
    <div class="main-content">
        <?php if ($message): ?>
            <div class="alert alert-<?php echo $messageType; ?> alert-dismissible fade show">
                <?php echo htmlspecialchars($message); ?>
                <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
            </div>
        <?php endif; ?>

        <div class="tab-content">
            <!-- Servers Tab -->
            <div class="tab-pane fade show active" id="servers">
                <div class="d-flex justify-content-between align-items-center mb-4">
                    <h2><i class="bi bi-hdd-stack me-2"></i>Xtream Servers</h2>
                    <?php if (count($servers) < MAX_SERVERS): ?>
                        <button class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#serverModal">
                            <i class="bi bi-plus-lg me-1"></i>Add Server
                        </button>
                    <?php endif; ?>
                </div>

                <div class="alert alert-info">
                    <i class="bi bi-info-circle me-2"></i>
                    Configure up to <?php echo MAX_SERVERS; ?> servers. The app will try them in priority order (1 = primary, 2-5 = fallbacks).
                </div>

                <?php if (empty($servers)): ?>
                    <div class="text-center py-5">
                        <i class="bi bi-hdd-stack" style="font-size: 48px; opacity: 0.5;"></i>
                        <p class="mt-3 text-muted">No servers configured yet</p>
                        <button class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#serverModal">
                            <i class="bi bi-plus-lg me-1"></i>Add Your First Server
                        </button>
                    </div>
                <?php else: ?>
                    <?php foreach ($servers as $server): ?>
                        <div class="server-card <?php echo $server['is_active'] ? 'active' : 'inactive'; ?>">
                            <div class="d-flex justify-content-between align-items-start">
                                <div>
                                    <span class="priority-badge me-2">#<?php echo $server['priority']; ?></span>
                                    <strong><?php echo htmlspecialchars($server['name']); ?></strong>
                                    <?php if (!$server['is_active']): ?>
                                        <span class="badge bg-secondary ms-2">Inactive</span>
                                    <?php endif; ?>
                                </div>
                                <div>
                                    <button class="btn btn-sm btn-outline-light me-1"
                                            onclick="editServer(<?php echo htmlspecialchars(json_encode($server)); ?>)">
                                        <i class="bi bi-pencil"></i>
                                    </button>
                                    <form method="POST" class="d-inline"
                                          onsubmit="return confirm('Delete this server?');">
                                        <input type="hidden" name="action" value="delete_server">
                                        <input type="hidden" name="server_id" value="<?php echo $server['id']; ?>">
                                        <button type="submit" class="btn btn-sm btn-outline-danger">
                                            <i class="bi bi-trash"></i>
                                        </button>
                                    </form>
                                </div>
                            </div>
                            <div class="mt-2 text-muted">
                                <code><?php echo $server['protocol']; ?>://<?php echo htmlspecialchars($server['host']); ?><?php echo ($server['port'] && $server['port'] !== '80' && $server['port'] !== '443') ? ':' . $server['port'] : ''; ?></code>
                            </div>
                        </div>
                    <?php endforeach; ?>
                <?php endif; ?>
            </div>

            <!-- Settings Tab -->
            <div class="tab-pane fade" id="settings">
                <h2 class="mb-4"><i class="bi bi-gear me-2"></i>App Settings</h2>

                <div class="card">
                    <div class="card-body">
                        <form method="POST">
                            <input type="hidden" name="action" value="save_settings">

                            <div class="mb-3">
                                <label class="form-label">App Name</label>
                                <input type="text" class="form-control" name="settings[app_name]"
                                       value="<?php echo htmlspecialchars($settings['app_name'] ?? 'LUXOR IPTV'); ?>">
                            </div>

                            <div class="mb-3">
                                <label class="form-label">Logo URL (optional)</label>
                                <input type="url" class="form-control" name="settings[app_logo_url]"
                                       value="<?php echo htmlspecialchars($settings['app_logo_url'] ?? ''); ?>"
                                       placeholder="https://example.com/logo.png">
                            </div>

                            <div class="mb-3">
                                <label class="form-label">Primary Color</label>
                                <div class="input-group">
                                    <input type="color" class="form-control form-control-color"
                                           name="settings[primary_color]"
                                           value="<?php echo htmlspecialchars($settings['primary_color'] ?? '#6200EE'); ?>">
                                    <input type="text" class="form-control"
                                           value="<?php echo htmlspecialchars($settings['primary_color'] ?? '#6200EE'); ?>"
                                           disabled>
                                </div>
                            </div>

                            <div class="mb-3">
                                <label class="form-label">Max Devices Per User</label>
                                <input type="number" class="form-control" name="settings[max_devices_per_user]"
                                       value="<?php echo htmlspecialchars($settings['max_devices_per_user'] ?? '3'); ?>"
                                       min="1" max="10">
                            </div>

                            <button type="submit" class="btn btn-primary">
                                <i class="bi bi-check-lg me-1"></i>Save Settings
                            </button>
                        </form>
                    </div>
                </div>
            </div>

            <!-- Statistics Tab -->
            <div class="tab-pane fade" id="stats">
                <h2 class="mb-4"><i class="bi bi-bar-chart me-2"></i>Statistics (Last 24h)</h2>

                <div class="row mb-4">
                    <div class="col-md-3">
                        <div class="stat-card">
                            <div class="stat-number"><?php echo number_format($stats['total_connections'] ?? 0); ?></div>
                            <div>Total Connections</div>
                        </div>
                    </div>
                    <div class="col-md-3">
                        <div class="stat-card secondary">
                            <div class="stat-number"><?php echo number_format($stats['unique_devices'] ?? 0); ?></div>
                            <div>Unique Devices</div>
                        </div>
                    </div>
                    <div class="col-md-3">
                        <div class="stat-card success">
                            <div class="stat-number"><?php echo number_format($stats['successful'] ?? 0); ?></div>
                            <div>Successful</div>
                        </div>
                    </div>
                    <div class="col-md-3">
                        <div class="stat-card warning">
                            <div class="stat-number"><?php echo number_format($stats['failed'] ?? 0); ?></div>
                            <div>Failed</div>
                        </div>
                    </div>
                </div>

                <div class="card">
                    <div class="card-header">Recent Connections</div>
                    <div class="card-body">
                        <?php
                        $recentLogs = db()->fetchAll("
                            SELECT cl.*, xs.name as server_name
                            FROM connection_logs cl
                            LEFT JOIN xtream_servers xs ON cl.server_used = xs.id
                            ORDER BY cl.created_at DESC
                            LIMIT 20
                        ");
                        ?>
                        <?php if (empty($recentLogs)): ?>
                            <p class="text-muted text-center py-3">No connection logs yet</p>
                        <?php else: ?>
                            <div class="table-responsive">
                                <table class="table table-dark table-sm">
                                    <thead>
                                        <tr>
                                            <th>Time</th>
                                            <th>Device ID</th>
                                            <th>Username</th>
                                            <th>Server</th>
                                            <th>IP</th>
                                            <th>Status</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        <?php foreach ($recentLogs as $log): ?>
                                            <tr>
                                                <td><?php echo date('M j, H:i', strtotime($log['created_at'])); ?></td>
                                                <td><code><?php echo htmlspecialchars(substr($log['device_id'] ?? '-', 0, 12)); ?>...</code></td>
                                                <td><?php echo htmlspecialchars($log['username'] ?? '-'); ?></td>
                                                <td><?php echo htmlspecialchars($log['server_name'] ?? '-'); ?></td>
                                                <td><?php echo htmlspecialchars($log['ip_address'] ?? '-'); ?></td>
                                                <td>
                                                    <span class="badge bg-<?php echo $log['status'] === 'success' ? 'success' : 'danger'; ?>">
                                                        <?php echo $log['status']; ?>
                                                    </span>
                                                </td>
                                            </tr>
                                        <?php endforeach; ?>
                                    </tbody>
                                </table>
                            </div>
                        <?php endif; ?>
                    </div>
                </div>
            </div>

            <!-- Account Tab -->
            <div class="tab-pane fade" id="account">
                <h2 class="mb-4"><i class="bi bi-person me-2"></i>Account Settings</h2>

                <div class="card">
                    <div class="card-header">Change Password</div>
                    <div class="card-body">
                        <form method="POST">
                            <input type="hidden" name="action" value="change_password">

                            <div class="mb-3">
                                <label class="form-label">Current Password</label>
                                <input type="password" class="form-control" name="current_password" required>
                            </div>

                            <div class="mb-3">
                                <label class="form-label">New Password</label>
                                <input type="password" class="form-control" name="new_password" required minlength="6">
                            </div>

                            <div class="mb-3">
                                <label class="form-label">Confirm New Password</label>
                                <input type="password" class="form-control" name="confirm_password" required>
                            </div>

                            <button type="submit" class="btn btn-primary">
                                <i class="bi bi-key me-1"></i>Change Password
                            </button>
                        </form>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Server Modal -->
    <div class="modal fade" id="serverModal" tabindex="-1">
        <div class="modal-dialog">
            <div class="modal-content bg-dark">
                <div class="modal-header border-secondary">
                    <h5 class="modal-title" id="serverModalTitle">Add Server</h5>
                    <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal"></button>
                </div>
                <form method="POST" id="serverForm">
                    <div class="modal-body">
                        <input type="hidden" name="action" value="save_server">
                        <input type="hidden" name="server_id" id="serverId">

                        <div class="mb-3">
                            <label class="form-label">Priority (1 = Primary)</label>
                            <select class="form-select" name="priority" id="serverPriority" required>
                                <?php for ($i = 1; $i <= MAX_SERVERS; $i++): ?>
                                    <option value="<?php echo $i; ?>"><?php echo $i; ?></option>
                                <?php endfor; ?>
                            </select>
                        </div>

                        <div class="mb-3">
                            <label class="form-label">Server Name</label>
                            <input type="text" class="form-control" name="name" id="serverName"
                                   placeholder="e.g., Primary Server" required>
                        </div>

                        <div class="mb-3">
                            <label class="form-label">Host</label>
                            <input type="text" class="form-control" name="host" id="serverHost"
                                   placeholder="e.g., iptv.example.com" required>
                        </div>

                        <div class="row">
                            <div class="col-6 mb-3">
                                <label class="form-label">Protocol</label>
                                <select class="form-select" name="protocol" id="serverProtocol">
                                    <option value="http">HTTP</option>
                                    <option value="https">HTTPS</option>
                                </select>
                            </div>
                            <div class="col-6 mb-3">
                                <label class="form-label">Port</label>
                                <input type="text" class="form-control" name="port" id="serverPort"
                                       placeholder="80" value="80">
                            </div>
                        </div>

                        <div class="form-check">
                            <input type="checkbox" class="form-check-input" name="is_active"
                                   id="serverActive" checked>
                            <label class="form-check-label" for="serverActive">Active</label>
                        </div>
                    </div>
                    <div class="modal-footer border-secondary">
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                        <button type="submit" class="btn btn-primary">Save Server</button>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        // Handle tab navigation from sidebar
        document.querySelectorAll('.sidebar .nav-link').forEach(link => {
            link.addEventListener('click', function(e) {
                document.querySelectorAll('.sidebar .nav-link').forEach(l => l.classList.remove('active'));
                this.classList.add('active');
            });
        });

        // Edit server function
        function editServer(server) {
            document.getElementById('serverModalTitle').textContent = 'Edit Server';
            document.getElementById('serverId').value = server.id;
            document.getElementById('serverPriority').value = server.priority;
            document.getElementById('serverName').value = server.name;
            document.getElementById('serverHost').value = server.host;
            document.getElementById('serverPort').value = server.port;
            document.getElementById('serverProtocol').value = server.protocol;
            document.getElementById('serverActive').checked = server.is_active == 1;

            new bootstrap.Modal(document.getElementById('serverModal')).show();
        }

        // Reset modal on close
        document.getElementById('serverModal').addEventListener('hidden.bs.modal', function() {
            document.getElementById('serverModalTitle').textContent = 'Add Server';
            document.getElementById('serverForm').reset();
            document.getElementById('serverId').value = '';
            document.getElementById('serverActive').checked = true;
        });
    </script>
</body>
</html>
