<?php
/**
 * LUXOR IPTV Admin Panel - Authentication
 */

require_once __DIR__ . '/db.php';

class Auth {

    /**
     * Attempt to login with username and password
     */
    public static function login($username, $password) {
        $user = db()->fetch(
            "SELECT * FROM admins WHERE username = :username AND is_active = 1",
            ['username' => $username]
        );

        if ($user && password_verify($password, $user['password'])) {
            $_SESSION['admin_id'] = $user['id'];
            $_SESSION['admin_username'] = $user['username'];
            $_SESSION['logged_in'] = true;

            // Update last login
            db()->update('admins',
                ['last_login' => date('Y-m-d H:i:s')],
                'id = :id',
                ['id' => $user['id']]
            );

            return true;
        }

        return false;
    }

    /**
     * Check if user is logged in
     */
    public static function check() {
        return isset($_SESSION['logged_in']) && $_SESSION['logged_in'] === true;
    }

    /**
     * Get current admin user
     */
    public static function user() {
        if (!self::check()) {
            return null;
        }

        return db()->fetch(
            "SELECT id, username, email, created_at, last_login FROM admins WHERE id = :id",
            ['id' => $_SESSION['admin_id']]
        );
    }

    /**
     * Logout
     */
    public static function logout() {
        $_SESSION = [];
        session_destroy();
    }

    /**
     * Require authentication - redirect to login if not authenticated
     */
    public static function require() {
        if (!self::check()) {
            header('Location: index.php');
            exit;
        }
    }

    /**
     * Change password
     */
    public static function changePassword($currentPassword, $newPassword) {
        if (!self::check()) {
            return false;
        }

        $user = db()->fetch(
            "SELECT * FROM admins WHERE id = :id",
            ['id' => $_SESSION['admin_id']]
        );

        if (!$user || !password_verify($currentPassword, $user['password'])) {
            return false;
        }

        $hashedPassword = password_hash($newPassword, PASSWORD_DEFAULT);
        db()->update('admins',
            ['password' => $hashedPassword],
            'id = :id',
            ['id' => $_SESSION['admin_id']]
        );

        return true;
    }

    /**
     * Hash password
     */
    public static function hashPassword($password) {
        return password_hash($password, PASSWORD_DEFAULT);
    }
}
